const { app } = require('electron');
const fs = require('fs');
const fsp = fs.promises;
const path = require('path');
const crypto = require('crypto');
const logger = require('../logger');

const STORE_FILENAME = 'oauth-tokens.json';
const STORE_VERSION = 2;
const LEGACY_ACCOUNT_PREFIX = 'legacy-account';

let storeCache = null;
let storeLoaded = false;
let storeLoadPromise = null;

function getStorePath() {
  const storageDir = app.getPath('userData');
  return path.join(storageDir, STORE_FILENAME);
}

function createEmptyStore() {
  return {
    version: STORE_VERSION,
    activeAccountId: null,
    accounts: {}
  };
}

function cloneTokens(tokens) {
  if (!tokens || typeof tokens !== 'object') {
    return {};
  }
  return { ...tokens };
}

function cloneAccount(account) {
  if (!account) {
    return null;
  }
  return {
    ...account,
    tokens: cloneTokens(account.tokens),
    scopes: Array.isArray(account.scopes) ? [...account.scopes] : [],
    createdAt: account.createdAt || null,
    updatedAt: account.updatedAt || null,
    lastUsedAt: account.lastUsedAt || null
  };
}

function isLegacyTokenShape(value) {
  if (!value || typeof value !== 'object') {
    return false;
  }
  const legacyKeys = ['access_token', 'refresh_token', 'scope', 'token_type', 'expiry_date'];
  return legacyKeys.some((key) => Object.prototype.hasOwnProperty.call(value, key));
}

function generateLegacyAccountId() {
  return `${LEGACY_ACCOUNT_PREFIX}-${crypto.randomBytes(6).toString('hex')}`;
}

function sanitizeAccountRecord(id, payload = {}) {
  if (!id) {
    id = generateLegacyAccountId();
  }

  const now = new Date().toISOString();
  const tokens = payload.tokens && typeof payload.tokens === 'object' ? payload.tokens : payload;
  const scopes = Array.isArray(payload.scopes) ? payload.scopes.filter(Boolean) : [];

  const record = {
    id,
    channelId: typeof payload.channelId === 'string' ? payload.channelId : null,
    channelTitle: typeof payload.channelTitle === 'string' ? payload.channelTitle : null,
    channelCustomUrl: typeof payload.channelCustomUrl === 'string' ? payload.channelCustomUrl : null,
    channelThumbnailUrl: typeof payload.channelThumbnailUrl === 'string' ? payload.channelThumbnailUrl : null,
    tokens: cloneTokens(tokens),
    scopes,
    createdAt: payload.createdAt || now,
    updatedAt: payload.updatedAt || now,
    lastUsedAt: payload.lastUsedAt || null
  };

  return record;
}

function sanitizeStore(rawStore) {
  const store = createEmptyStore();
  if (!rawStore || typeof rawStore !== 'object') {
    return store;
  }

  const entries = rawStore.accounts && typeof rawStore.accounts === 'object'
    ? Object.entries(rawStore.accounts)
    : [];

  for (const [key, value] of entries) {
    if (!value || typeof value !== 'object') {
      continue;
    }
    const id = typeof value.id === 'string' && value.id.trim().length > 0 ? value.id : key;
    const record = sanitizeAccountRecord(id, value);
    store.accounts[record.id] = record;
  }

  const accountIds = Object.keys(store.accounts);
  const preferredActive = typeof rawStore.activeAccountId === 'string' ? rawStore.activeAccountId : null;
  store.activeAccountId = preferredActive && store.accounts[preferredActive]
    ? preferredActive
    : (accountIds.length > 0 ? accountIds[0] : null);
  store.version = Number.isFinite(rawStore.version) ? rawStore.version : STORE_VERSION;

  return store;
}

async function readStoreFile() {
  const storePath = getStorePath();
  try {
    const raw = await fsp.readFile(storePath, 'utf8');
    return JSON.parse(raw);
  } catch (error) {
    if (error && error.code !== 'ENOENT') {
      logger.warn('Failed to read YouTube token store:', error);
    }
    return null;
  }
}

async function loadStoreFromDisk() {
  const raw = await readStoreFile();
  if (raw && raw.accounts) {
    return sanitizeStore(raw);
  }

  if (isLegacyTokenShape(raw)) {
    const accountId = typeof raw.channelId === 'string' && raw.channelId.length > 0
      ? raw.channelId
      : generateLegacyAccountId();
    const record = sanitizeAccountRecord(accountId, raw);
    const store = createEmptyStore();
    store.accounts[record.id] = {
      ...record,
      createdAt: record.createdAt,
      updatedAt: record.updatedAt,
      lastUsedAt: record.lastUsedAt || record.updatedAt
    };
    store.activeAccountId = record.id;
    return store;
  }

  return createEmptyStore();
}

async function ensureStoreLoaded() {
  if (storeLoaded) {
    return;
  }
  if (!storeLoadPromise) {
    storeLoadPromise = (async () => {
      storeCache = await loadStoreFromDisk();
      storeLoaded = true;
      return storeCache;
    })();
  }
  await storeLoadPromise;
}

async function saveStore() {
  await ensureStoreLoaded();
  const storePath = getStorePath();
  const directory = path.dirname(storePath);
  const payload = {
    ...storeCache,
    version: STORE_VERSION
  };
  await fsp.mkdir(directory, { recursive: true });
  await fsp.writeFile(storePath, JSON.stringify(payload, null, 2), 'utf8');
}

async function getStoreSnapshot() {
  await ensureStoreLoaded();
  return {
    version: storeCache.version,
    activeAccountId: storeCache.activeAccountId,
    accounts: Object.fromEntries(
      Object.entries(storeCache.accounts).map(([id, account]) => [id, cloneAccount(account)])
    )
  };
}

async function listAccounts() {
  await ensureStoreLoaded();
  return Object.values(storeCache.accounts).map((account) => cloneAccount(account));
}

async function getAccount(accountId) {
  await ensureStoreLoaded();
  if (!accountId) {
    return null;
  }
  return cloneAccount(storeCache.accounts[accountId] || null);
}

async function getActiveAccountId() {
  await ensureStoreLoaded();
  return storeCache.activeAccountId || null;
}

async function setActiveAccount(accountId) {
  await ensureStoreLoaded();
  if (accountId && !storeCache.accounts[accountId]) {
    throw new Error(`Cannot set active YouTube account. Account "${accountId}" does not exist.`);
  }
  storeCache.activeAccountId = accountId || null;
  await saveStore();
  return storeCache.activeAccountId;
}

async function ensureActiveAccount() {
  await ensureStoreLoaded();
  if (storeCache.activeAccountId && storeCache.accounts[storeCache.activeAccountId]) {
    return storeCache.activeAccountId;
  }
  const nextId = Object.keys(storeCache.accounts)[0] || null;
  storeCache.activeAccountId = nextId;
  if (storeLoaded) {
    await saveStore();
  }
  return nextId;
}

async function upsertAccount(accountPayload) {
  await ensureStoreLoaded();
  if (!accountPayload || typeof accountPayload !== 'object') {
    throw new Error('Invalid account payload.');
  }

  const now = new Date().toISOString();
  const base = sanitizeAccountRecord(accountPayload.id, accountPayload);
  const existing = storeCache.accounts[base.id] || null;

  const nextRecord = {
    ...existing,
    ...base,
    tokens: cloneTokens(base.tokens),
    scopes: Array.isArray(base.scopes) ? [...new Set(base.scopes.filter(Boolean))] : [],
    createdAt: existing && existing.createdAt ? existing.createdAt : (base.createdAt || now),
    updatedAt: now,
    lastUsedAt: base.lastUsedAt || now
  };

  storeCache.accounts[nextRecord.id] = nextRecord;
  if (!storeCache.activeAccountId) {
    storeCache.activeAccountId = nextRecord.id;
  }
  await saveStore();
  return cloneAccount(nextRecord);
}

async function mergeAccountTokens(accountId, tokens) {
  await ensureStoreLoaded();
  if (!accountId) {
    throw new Error('Account ID is required to merge tokens.');
  }
  const account = storeCache.accounts[accountId];
  if (!account) {
    throw new Error(`Cannot merge tokens. Account "${accountId}" not found.`);
  }
  if (!tokens || typeof tokens !== 'object') {
    return cloneAccount(account);
  }
  account.tokens = {
    ...account.tokens,
    ...cloneTokens(tokens)
  };
  account.updatedAt = new Date().toISOString();
  await saveStore();
  return cloneAccount(account);
}

async function removeAccount(accountId) {
  await ensureStoreLoaded();
  if (!accountId || !storeCache.accounts[accountId]) {
    return false;
  }
  delete storeCache.accounts[accountId];
  if (storeCache.activeAccountId === accountId) {
    const remainingIds = Object.keys(storeCache.accounts);
    storeCache.activeAccountId = remainingIds.length > 0 ? remainingIds[0] : null;
  }
  await saveStore();
  return true;
}

async function clearStore() {
  storeCache = createEmptyStore();
  storeLoaded = true;
  storeLoadPromise = null;
  await saveStore();
}

async function markAccountUsed(accountId) {
  await ensureStoreLoaded();
  if (!accountId) {
    return;
  }
  const account = storeCache.accounts[accountId];
  if (!account) {
    return;
  }
  account.lastUsedAt = new Date().toISOString();
  await saveStore();
}

async function getAccountTokens(accountId) {
  await ensureStoreLoaded();
  if (!accountId) {
    accountId = await ensureActiveAccount();
  }
  if (!accountId) {
    return null;
  }
  const account = storeCache.accounts[accountId];
  if (!account) {
    return null;
  }
  return cloneTokens(account.tokens);
}

module.exports = {
  getStorePath,
  getStoreSnapshot,
  listAccounts,
  getAccount,
  getAccountTokens,
  getActiveAccountId,
  setActiveAccount,
  upsertAccount,
  mergeAccountTokens,
  removeAccount,
  clearStore,
  markAccountUsed
};

